#!/bin/bash

# Set strict mode
set -euo pipefail

# Global variables
LOG_FILE="/tmp/fastapi_setup.log"
REQUIRED_PACKAGES=("python3" "python3-pip" "python3-venv" "nginx" "git" "net-tools" "certbot" "python3-certbot-nginx")

# Create log directory if it doesn't exist
touch "$LOG_FILE"

# Function to log messages
log_message() {
    echo "$(date '+%Y-%m-%d %H:%M:%S') - $1" | tee -a "$LOG_FILE"
}

# Function to prompt yes/no questions
prompt_yes_no() {
    while true; do
        read -rp "$1 (y/n): " yn
        case $yn in
            [Yy]* ) return 0;;
            [Nn]* ) return 1;;
            * ) echo "Please answer yes (y) or no (n).";;
        esac
    done
}

# Function to check if a port is in use
is_port_in_use() {
    if netstat -tuln | grep -q ":$1 "; then
        return 0
    else
        return 1
    fi
}

# Function to check if a service exists
service_exists() {
    if systemctl list-unit-files | grep -q "^$1.service"; then
        return 0
    else
        return 1
    fi
}

# Function to check if nginx config exists
nginx_config_exists() {
    if [ -f "/etc/nginx/sites-available/$1" ] || [ -f "/etc/nginx/sites-enabled/$1" ]; then
        return 0
    else
        return 1
    fi
}

# Function to install system dependencies
install_dependencies() {
    log_message "Checking for system updates..."
    if prompt_yes_no "Do you want to update system packages?"; then
        sudo apt-get update
        sudo apt-get upgrade -y
    fi

    log_message "Installing required packages..."
    for package in "${REQUIRED_PACKAGES[@]}"; do
        if ! dpkg -l | grep -q "^ii  $package"; then
            log_message "Installing $package..."
            sudo apt-get install -y "$package"
        else
            log_message "Package $package is already installed"
        fi
    done
}

# Function to setup SSL with Let's Encrypt
setup_ssl() {
    local domain=$1
    local email=$2
    log_message "Setting up SSL for $domain using email $email..."
    
    if ! sudo certbot --nginx -d "$domain" --non-interactive --agree-tos --email "$email" --redirect; then
        log_message "Failed to setup SSL. Check certbot logs for details."
        return 1
    fi
    
    log_message "SSL setup completed successfully"
    return 0
}

# Function to check service status
check_service_status() {
    local service_name=$1
    local max_attempts=5
    local attempt=1
    
    log_message "Checking service status..."
    while [ $attempt -le $max_attempts ]; do
        if systemctl is-active --quiet "$service_name"; then
            log_message "Service $service_name is running successfully"
            # Check logs for any errors
            log_message "Recent service logs:"
            journalctl -u "$service_name" -n 50 --no-pager | tee -a "$LOG_FILE"
            return 0
        else
            log_message "Attempt $attempt/$max_attempts: Service is not running properly"
            log_message "Service status:"
            systemctl status "$service_name" | tee -a "$LOG_FILE"
            log_message "Service logs:"
            journalctl -u "$service_name" -n 50 --no-pager | tee -a "$LOG_FILE"
            
            if [ $attempt -lt $max_attempts ]; then
                log_message "Waiting 10 seconds before next check..."
                sleep 10
            fi
        fi
        attempt=$((attempt + 1))
    done
    
    log_message "Service failed to start properly after $max_attempts attempts"
    return 1
}

# Function to create nginx configuration
create_nginx_config() {
    local domain=$1
    local port=$2
    local config_file="/etc/nginx/sites-available/$domain"

    cat > "$config_file" <<EOF
server {
    listen 80;
    listen [::]:80;
    server_name $domain;

    location / {
        proxy_pass http://localhost:$port;
        proxy_http_version 1.1;
        proxy_set_header Upgrade \$http_upgrade;
        proxy_set_header Connection "upgrade";
        proxy_set_header Host \$host;
        proxy_set_header X-Real-IP \$remote_addr;
        proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;
        proxy_set_header X-Forwarded-Proto \$scheme;
    }
}
EOF

    sudo ln -sf "$config_file" "/etc/nginx/sites-enabled/"
    sudo nginx -t && sudo systemctl reload nginx
}

# Function to create systemd service
create_systemd_service() {
    local service_name=$1
    local app_dir=$2
    local port=$3
    local workers=$4
    local app_path=$5

    cat > "/etc/systemd/system/$service_name.service" <<EOF
[Unit]
Description=FastAPI application service
After=network.target

[Service]
User=$USER
Group=$USER
WorkingDirectory=$app_dir
Environment="PATH=$app_dir/venv/bin:\$PATH"
ExecStart=$app_dir/venv/bin/uvicorn $app_path --host 0.0.0.0 --port $port --workers $workers
Restart=always

[Install]
WantedBy=multi-user.target
EOF

    sudo systemctl daemon-reload
    sudo systemctl enable "$service_name"
    sudo systemctl start "$service_name"
}

# Main setup function
main() {
    # Welcome message
    echo "FastAPI Deployment Script"
    echo "========================"

    # Install dependencies first
    log_message "Installing system dependencies..."
    install_dependencies

    # Collect user inputs
    read -rp "Enter Git repository URL: " REPO_URL
    read -rp "Enter Git token: " GIT_TOKEN
    read -rp "Enter target directory path: " TARGET_DIR
    read -rp "Enter service name: " SERVICE_NAME
    read -rp "Enter domain name: " DOMAIN_NAME
    read -rp "Enter port number (default 8000): " PORT_NUMBER
    PORT_NUMBER=${PORT_NUMBER:-8000}
    read -rp "Enter number of workers (default 1): " WORKERS
    WORKERS=${WORKERS:-1}
    read -rp "Enter FastAPI application path (default main:app): " APP_PATH
    APP_PATH=${APP_PATH:-main:app}
    
    echo "Let's Encrypt requires an email address for important notifications about:"
    echo "- SSL certificate expiration"
    echo "- Security notices"
    echo "- Account recovery"
    read -rp "Enter email for SSL notifications: " SSL_EMAIL

    # Validate service and nginx config before proceeding
    if service_exists "$SERVICE_NAME"; then
        log_message "Error: Service $SERVICE_NAME already exists"
        exit 1
    fi

    if nginx_config_exists "$DOMAIN_NAME"; then
        log_message "Error: Nginx configuration for $DOMAIN_NAME already exists"
        exit 1
    fi

    if is_port_in_use "$PORT_NUMBER"; then
        log_message "Error: Port $PORT_NUMBER is already in use"
        exit 1
    fi

    # Create a temporary directory for cloning
    TEMP_DIR=$(mktemp -d)
    log_message "Created temporary directory for cloning: $TEMP_DIR"
    
    # Clone repository into temp directory
    log_message "Cloning repository..."
    
    # Validate and format repository URL
    if [[ ! "$REPO_URL" =~ ^https://github\.com.* ]]; then
        log_message "Error: Repository URL must start with 'https://github.com/'"
        log_message "Example: https://github.com/username/repository"
        exit 1
    fi
    
    # Format the URL with token
    REPO_URL_WITHOUT_HTTPS=${REPO_URL#https://}
    AUTH_REPO_URL="https://${GIT_TOKEN}@${REPO_URL_WITHOUT_HTTPS}"
    
    # Attempt to clone with detailed error handling
    if ! git clone "$AUTH_REPO_URL" "$TEMP_DIR" 2> git_error.log; then
        ERROR_MSG=$(cat git_error.log)
        log_message "Git clone failed with error:"
        log_message "$ERROR_MSG"
        log_message "Please ensure:"
        log_message "1. The repository URL is correct (should be like https://github.com/username/repo)"
        log_message "2. The access token has correct permissions (repo access)"
        log_message "3. The token is valid and not expired"
        rm git_error.log
        exit 1
    fi
    
    log_message "Repository cloned successfully"
    
    # Prepare target directory
    if [ -d "$TARGET_DIR" ]; then
        log_message "Cleaning target directory..."
        sudo rm -rf "${TARGET_DIR:?}"
    fi
    
    # Create fresh target directory and set permissions
    sudo mkdir -p "$TARGET_DIR"
    sudo chown "$USER:$USER" "$TARGET_DIR"
    sudo chmod 755 "$TARGET_DIR"
    
    # Move contents from temp to target
    log_message "Moving files to target directory..."
    sudo cp -a "$TEMP_DIR/." "$TARGET_DIR/"
    sudo rm -rf "$TEMP_DIR"
    
    # Change to target directory
    cd "$TARGET_DIR" || exit 1

    # Create virtual environment
    log_message "Creating virtual environment..."
    python3 -m venv venv
    source venv/bin/activate

    # Install requirements with memory optimization
    if [ -f "requirements.txt" ]; then
        log_message "Installing requirements (this may take a while)..."
        # Clear pip cache first
        pip cache purge
        
        # Install with no cache and show progress
        if ! pip install --no-cache-dir -r requirements.txt 2> pip_error.log; then
            ERROR_MSG=$(cat pip_error.log)
            log_message "Failed to install requirements:"
            log_message "$ERROR_MSG"
            log_message "This might be due to memory constraints. Try:"
            log_message "1. Clearing system cache: sudo sh -c 'echo 3 > /proc/sys/vm/drop_caches'"
            log_message "2. Adding swap space if needed"
            rm pip_error.log
            exit 1
        fi
        log_message "Requirements installed successfully"
    else
        log_message "Warning: requirements.txt not found"
    fi

    # Install uvicorn with memory optimization
    log_message "Installing uvicorn..."
    if ! pip install --no-cache-dir uvicorn 2> pip_error.log; then
        ERROR_MSG=$(cat pip_error.log)
        log_message "Failed to install uvicorn:"
        log_message "$ERROR_MSG"
        rm pip_error.log
        exit 1
    fi

    # Configure nginx
    if prompt_yes_no "Do you want to configure nginx?"; then
        log_message "Configuring nginx..."
        create_nginx_config "$DOMAIN_NAME" "$PORT_NUMBER"
    fi

    # Create systemd service
    if prompt_yes_no "Do you want to create and start the systemd service?"; then
        log_message "Creating systemd service..."
        create_systemd_service "$SERVICE_NAME" "$TARGET_DIR" "$PORT_NUMBER" "$WORKERS" "$APP_PATH"
    fi

    # Deactivate virtual environment
    deactivate

    # Setup SSL
    if prompt_yes_no "Do you want to setup SSL with Let's Encrypt?"; then
        setup_ssl "$DOMAIN_NAME" "$SSL_EMAIL"
    fi

    # Check service status
    log_message "Checking if service is running properly..."
    if ! check_service_status "$SERVICE_NAME"; then
        log_message "WARNING: Service may not be running correctly. Check the logs above for details."
    fi

    # Final messages
    log_message "Installation completed successfully!"
    echo "IMPORTANT: Don't forget to:"
    echo "1. Create .env file if needed"
    echo "2. Check logs at $LOG_FILE for any issues"
    echo "3. Use your update script for future updates"
    echo "4. Your API should now be accessible at: https://$DOMAIN_NAME"
}

# Run main function
main
