#!/bin/bash
# Set strict mode
set -euo pipefail

# Configuration
REPO_URL=""
REPO_DIR=""
SERVICE_NAME=""
LOG_FILE="$(dirname "$0")/update_log.txt"
GIT_TOKEN=""
VENV_DIR="$REPO_DIR/venv"
REQUIREMENTS_HASH_FILE="$REPO_DIR/.requirements.md5"

# Function to log messages
log_message() {
    echo "$(date '+%Y-%m-%d %H:%M:%S') - $1" | tee -a "$LOG_FILE"
}

# Function to check if requirements have changed
check_requirements_changed() {
    if [ ! -f "$REQUIREMENTS_HASH_FILE" ]; then
        return 0  # Return true if hash file doesn't exist
    fi
    
    local current_hash
    local stored_hash
    
    current_hash=$(md5sum requirements.txt | awk '{print $1}')
    stored_hash=$(cat "$REQUIREMENTS_HASH_FILE")
    
    [ "$current_hash" != "$stored_hash" ]
}

# Function to update requirements hash
update_requirements_hash() {
    md5sum requirements.txt | awk '{print $1}' > "$REQUIREMENTS_HASH_FILE"
}

# Start logging
log_message "Starting update process"

# Ensure we're in the correct directory
cd "$REPO_DIR" || { log_message "Error: Unable to change to repository directory"; exit 1; }

# Construct the URL with the token
AUTH_REPO_URL="https://${GIT_TOKEN}@github.com/hassancs91/promoterkit-api.git"

# Fetch latest changes
log_message "Fetching latest changes from repository"
if ! git fetch "$AUTH_REPO_URL" main; then
    log_message "Error: Failed to fetch from repository"
    exit 1
fi

# Get the latest commit hash from the remote main branch
REMOTE=$(git rev-parse FETCH_HEAD)
# Get the latest commit hash from the local branch
LOCAL=$(git rev-parse HEAD)

log_message "Local commit: $LOCAL"
log_message "Remote commit: $REMOTE"

if [ "$LOCAL" != "$REMOTE" ]; then
    log_message "Updates available, pulling changes"
    if ! git pull "$AUTH_REPO_URL" main; then
        log_message "Error: Failed to pull changes"
        exit 1
    fi

    # Check if requirements have changed
    if check_requirements_changed; then
        log_message "Requirements changes detected, updating dependencies"
        
        # Activate virtual environment
        log_message "Activating virtual environment"
        if [ ! -d "$VENV_DIR" ]; then
            log_message "Error: Virtual environment not found at $VENV_DIR"
            exit 1
        fi
        source "$VENV_DIR/bin/activate" || { log_message "Error: Failed to activate virtual environment"; exit 1; }
        
        # Update dependencies
        if ! pip install -r requirements.txt; then
            log_message "Error: Failed to update dependencies"
            deactivate
            exit 1
        fi
        
        # Update requirements hash
        update_requirements_hash
        log_message "Dependencies updated successfully"
        
        # Deactivate virtual environment
        deactivate
    else
        log_message "No changes in requirements.txt, skipping dependency update"
    fi

    # Restart the service
    log_message "Restarting FastAPI service"
    if ! sudo systemctl restart "$SERVICE_NAME"; then
        log_message "Error: Failed to restart service"
        exit 1
    fi
    log_message "Service updated and restarted successfully"
else
    log_message "No updates available"
fi

# Clean up: remove the token from Git config
git config --unset-all url."https://${GIT_TOKEN}@github.com/".insteadOf
log_message "Update process completed"